/*
 * StraightExactMovementInfo.h
 *
 * Created 8/31/2009 By Johnny Huynh
 *
 * Version 00.00.01 8/31/2009
 *
 * Copyright Information:
 * All content copyright  2009 Johnny Huynh. All rights reserved.
 */
 
 /**
  * Contains information for an exact straight movement to a specified location.
  */
 
 #ifndef STRAIGHT_EXACT_MOVEMENT_INFO_H
 #define STRAIGHT_EXACT_MOVEMENT_INFO_H
 
 template <typename T> class StraightExactMovementInfo;
 
 #include "global.h"
 
 #include "MovementInfo.h"
 
 /**
  * Class specification for StraightExactMovementInfo
  */
 template <typename T>
 class StraightExactMovementInfo : public StraightMovementInfo<T>
 {
 // Data Members
 private:
    VECTOR3_TYPE _begin_point; // geographical coordinates vector representing a location relative to the Object
    
 // Local Functions
 public:
    StraightExactMovementInfo( const VECTOR3_TYPE& begin_point = VECTOR3_TYPE( ZERO, ZERO, ZERO ), 
                               const VECTOR3_TYPE& direction = VECTOR3_TYPE( ZERO, ZERO, ZERO ),
                               const T& distance = ZERO, const double duration = 0.0 );
    StraightExactMovementInfo( const StraightExactMovementInfo<T>& straight_exact_movement_info );
    virtual ~StraightExactMovementInfo();
    inline StraightExactMovementInfo<T>& operator=( const StraightExactMovementInfo<T>& straight_exact_movement_info );
    inline VECTOR3_TYPE get_begin_point() const;
    inline void set_begin_point( const VECTOR3_TYPE& begin_point );
    virtual inline double process_movement( Object<T>* obj_Ptr, MoveInfo<T>* move_info_Ptr, double current_time );
 
 // Private Functions
 private:
    
 // Public Static Functions
 public:
    
 };
 
 /** LOCAL FUNCTIONS **/
 
 /**
  * Constructor
  */
 template <typename T>
 StraightExactMovementInfo<T>::StraightExactMovementInfo( const VECTOR3_TYPE& begin_point, 
                                                          const VECTOR3_TYPE& direction,
                                                          const T& distance, const double duration )
                              : StraightMovementInfo<T>( direction, distance, duration ),
                                _begin_point( begin_point )
 {
 
 }
 
 /**
  * Copy Constructor
  */
 template <typename T>
 StraightExactMovementInfo<T>::StraightExactMovementInfo( const StraightExactMovementInfo<T>& straight_exact_movement_info )
                              : StraightMovementInfo<T>( straight_exact_movement_info ),
                                _begin_point( straight_exact_movement_info._begin_point )
 {
    
 }
 
 /**
  * Destructor
  */
 template <typename T>
 StraightExactMovementInfo<T>::~StraightExactMovementInfo()
 {
    
 }
 
 /**
  * operator=() copies the content of the specified StraightExactMovementInfo to this StraightExactMovementInfo.
  *
  * @param (const StraightExactMovementInfo<T>&) straight_exact_movement_info
  * @return StraightExactMovementInfo<T>&
  */
 template <typename T>
 inline StraightExactMovementInfo<T>& StraightExactMovementInfo<T>::operator=( const StraightExactMovementInfo<T>& straight_exact_movement_info )
 {
    StraightMovementInfo<T>::operator=( straight_exact_movement_info );
    _begin_point = straight_exact_movement_info._begin_point;
    
    return *this;
 }
 
 /**
  * get_begin_point() returns the straight movement starting location.
  *
  * @return VECTOR3_TYPE
  */
 template <typename T>
 inline VECTOR3_TYPE StraightExactMovementInfo<T>::get_begin_point() const
 {
    return _begin_point;
 }
 
 /**
  * set_begin_point() sets the straight movement starting location to the begin point.
  *
  * @param (const VECTOR3_TYPE&) begin_point
  */
 template <typename T>
 inline void StraightExactMovementInfo<T>::set_begin_point( const VECTOR3_TYPE& begin_point )
 {
    _begin_point = begin_point;
 }
 
 /**
  * process_movement() processes the movement specified by this MovementInfo, given the Object to process on,
  * the MoveInfo, and current time. The duration unprocessed returned is greater than zero if this MovementInfo
  * has finished processing, and there is time leftover.
  *
  * @param (Object<T>*) obj_Ptr
  * @param (MoveInfo<T>*) move_info_Ptr
  * @param (double) current_time
  * @return double - the time left unprocessed (i.e. a double greater than zero if this MovementInfo is done processing)
  */
 template <typename T>
 inline double StraightExactMovementInfo<T>::process_movement( Object<T>* obj_Ptr, MoveInfo<T>* move_info_Ptr, double current_time )
 {  
    nassertr( obj_Ptr != NULL, MovementInfo<T>::get_duration() );
    
    double move_action_invoke_time( move_info_Ptr->get_time_move_action_was_invoked() );
    double last_process_time( move_info_Ptr->get_time_last_processed_move_task() );
    nassertr( last_process_time >= move_action_invoke_time, MovementInfo<T>::get_duration() );
    
    double duration( MovementInfo<T>::get_duration() );
    double elapse_time_since_invocation( current_time - move_action_invoke_time );
    
    // gradually move for only the movement duration (measured in seconds)
    double elapse_time_since_last_movement( current_time - last_process_time );
    if ( elapse_time_since_invocation > duration ) // if true, this will be the last time we process this movement_info
        elapse_time_since_last_movement = duration + move_action_invoke_time - last_process_time;
    
    if ( elapse_time_since_last_movement > 0.0 && StraightMovementInfo<T>::get_direction() != VECTOR3_TYPE(ZERO,ZERO,ZERO) )
    {
        double invocation_ratio( elapse_time_since_invocation / duration );
        
        // move the Object to a specific location
        obj_Ptr->set_pos( (StraightMovementInfo<T>::get_direction() * invocation_ratio) + _begin_point );
    }
    
    if ( elapse_time_since_invocation > duration )
        return elapse_time_since_invocation - duration;
    else // elapse_time_since_invocation <= duration
        return 0.0;
 }
 
 /** PUBLIC STATIC FUNCTIONS **/
 
 #endif // STRAIGHT_EXACT_MOVEMENT_INFO_H